<?php

/**
 * Hook Class for Stock Sync with Google Sheet for WooCommerce Ultimate
 *
 * @package StockSyncWithGoogleSheetForWooCommerceUltimate
 * @since 1.0.0
 */

// Namespace.
namespace StockSyncWithGoogleSheetForWooCommerceUltimate;
use StockSyncWithGoogleSheetForWooCommerce\Sheet;
use StockSyncWithGoogleSheetForWooCommerce\Product;
// Exit if accessed directly.
defined( 'ABSPATH' ) || exit( 1 );

if ( ! class_exists( '\StockSyncWithGoogleSheetForWooCommerceUltimate\Hooks' ) ) {

	/**
	 * Hook Class for Stock Sync with Google Sheet for WooCommerce Ultimate
	 *
	 * @package StockSyncWithGoogleSheetForWooCommerceUltimate
	 * @since 1.0.0
	 */
	final class Hooks {

		/**
		 * Is license active
		 *
		 * @var bool
		 */
		protected $license_active;

		/**
		 * App class
		 *
		 * @var App
		 */
		public $app;

		/**
		 * Contains license information.
		 *
		 * @var mixed
		 */
		public $license;

		/**
		 * Constructor
		 */
		public function __construct() {
			$this->app = new App();
			$this->add_actions();
			$this->add_filters();
		}


		/**
		 * Add actions
		 *
		 * @return void
		 */
		public function add_actions() {

			$this->init_appsero_client();

			register_activation_hook( SSGSW_ULTIMATE, [ $this, 'activate' ] );
			register_deactivation_hook( SSGSW_ULTIMATE, [ $this, 'deactivate' ] );

			add_action( 'pre_current_active_plugins', [ $this, 'admin_notices' ] );

			add_action( 'init', [ $this, 'trigger_sync' ], 9999 );

			add_action( 'ssgsw_ultimate_triggered', [ $this, 'ssgsw_ultimate_triggered' ] );
			add_action( 'ssgsw_license_active_trigger', [ $this, 'ssgsw_ultimate_triggered' ] );
			add_action( 'ssgsw_license_deactive_trigger', [ $this, 'ssgsw_ultimate_triggered_false' ] );
		}
		/**
		 * Add filters
		 *
		 * @return void
		 */
		public function add_filters() {

			add_filter( 'is_ssgsw_license_active', $this->license_active ? '__return_true' : '__return_false', 0, 1 );

			add_filter( 'ssgsw_product_limit', [ $this, 'product_limit' ] );

			add_filter( 'ssgsw_columns', [ $this, 'product_columns' ] );

			add_filter( 'ssgsw_add_products_from_sheet', [ $this, 'add_products_from_sheet' ] );

			add_filter( 'ssgsw_queryable_metas', [ $this, 'queryable_metas' ] );
			add_filter('ssgsw_add_custom_meta_fileds', [$this, 'ssgsw_add_custom_meta_fileds']);
			add_filter('ssgsw_format_columns_data', [$this, 'ssgsw_format_columns_data'], 10, 3);
		}
		/**
		 * Format image and formula data
		 *
		 * @param array $format_data data.
		 *
		 * @return array formatted data.
		 */
		public function ssgsw_format_columns_data($formatted_data, $sheets_info, $formula) {
			if ($this->app->is_plugin_activated()) {
				$check_forumla_active = get_option('ssgsw_formula_active', false);
				if ( $check_forumla_active && $this->is_license_valid() ) {
					if ( empty($sheets_info) && false === $formula ) {
						$sheet = new Sheet();
						$sheets_info = $sheet->get_formula_value();
					}
					$sheet_header = isset($sheets_info[0]) ? $sheets_info[0] : [];
					$check_id_exist = isset($sheets_info[0][0]) && 'ID' === $sheets_info[0][0] ? true : false;
					$formatted_data = array_map(function ( $row ) use ( $sheets_info, $sheet_header, $check_id_exist ) { //phpcs:ignore
						$desired_value = $row['ID'];
						$found_array = null;
						if ( is_array($sheets_info) && ! empty($sheets_info) ) {
							foreach ( $sheets_info as $key => $sub_array ) {
								if ( isset($sub_array[0]) && $sub_array[0] === $desired_value ) {
									$found_array = $sub_array;
									break;
								}
							}
						}
	
						if ( is_array($found_array) ) {
							$i = 0;
							if ( $check_id_exist ) {
								$google_with_header = $this->set_sheet_header_with_data($sheet_header, $found_array);
								$database_data = $this->ssgsw_compare_with_formula_value($desired_value);
								$check_have_change = false;
								if ( ! empty($database_data) ) {
									$previous = [];
									foreach ( $row as $key => $value ) {
										$formation_key = $this->ssgsw_sheet_header_formating2($key);
										if ( isset($google_with_header[ $key ]) && strpos($google_with_header[ $key ], '=') === 0 ) {
											if ( 'product_image' !== $key ) {
												$row[ $key ] = $google_with_header[ $key ];
												$previous[0][ $formation_key ] = $database_data[ $formation_key ];
												if ( $value !== $database_data[ $formation_key ] ) {
													$check_have_change = true;
												}
											}
										} else {
											$previous[0][ $formation_key ] = $row[ $key ];
										}
										$i++;
									}
									if ( $check_have_change ) {
										$product_object = new Product();
										$product_object->bulk_update($previous, false, 1, false );
									}
								} else {
									foreach ( $row as $key => $value ) {
										if ( isset($google_with_header[ $key ]) && strpos($google_with_header[ $key ], '=') === 0 ) {
											if ( 'product_image' !== $key ) {
												$row[ $key ] = $google_with_header[ $key ];
											}
										}
										$i++;
									}
								}
							} else {
								foreach ( $row as $key => $value ) {
									if ( isset($found_array[ $i ]) && strpos($found_array[ $i ], '=') === 0 ) {
										if ( 'product_image' !== $key ) {
											$row[ $key ] = $found_array[ $i ];
										}
									}
									$i++;
								}
							}
						}
						if ( is_array($row) && array_key_exists('ssgsw_product_url', $row)) {
							$row['ssgsw_product_url'] = $this->generate_product_url($row);
						}
						if ( array_key_exists( 'product_image', $row) && ! empty($row['product_image']) ) {
							$image_url_convart = $row['product_image'];
							if ($image_url_convart) {
								$base_url = wp_upload_dir()['baseurl'];
								$absolute_url = trailingslashit($base_url) . ltrim($image_url_convart, '/');
								$row['product_image'] = '=IMAGE("' . $absolute_url . '")';
							} else {
								$row['product_image'] = '';
							}
							return array_values( (array) $row);
						} else {
							return array_values( (array) $row);
						}
					}, $formatted_data);
					
				} else {
					$formatted_data = array_map(function ( $row ) { //phpcs:ignore
						if ( is_array($row) && array_key_exists('ssgsw_product_url', $row)) {
							$row['ssgsw_product_url'] = $this->generate_product_url($row);
						}
						if ( array_key_exists( 'product_image', $row) && ! empty($row['product_image']) ) {
							$image_url_convart = $row['product_image'];
							if ($image_url_convart) {
								$base_url = wp_upload_dir()['baseurl'];
								$absolute_url = trailingslashit($base_url) . ltrim($image_url_convart, '/');
								$row['product_image'] = '=IMAGE("' . $absolute_url . '")';
							} else {
								$row['product_image'] = '';
							}
							return array_values( (array) $row);
						} else {
							return array_values( (array) $row);
						}
					}, $formatted_data);
					
				}
			}
			return $formatted_data;
		}
		/**
		 * Generate Product URL
		 */
		public function generate_product_url($row) {
			$p_id = $row['ID'];
			return get_permalink($p_id);
		}
		/**
		 * Retrieve parent id for product
		 *
		 * @param int $product_id product id.
		 *
		 * @return array product IDs
		 */
		public function get_parent_id( $child_id ) {
			global $wpdb;
			$child_product_ids = $wpdb->get_col(
				$wpdb->prepare(
					"
					SELECT post_parent
					FROM {$wpdb->posts}
					WHERE ID = %d
					AND (post_type = 'product' OR post_type = 'product_variation')
					",
					$child_id
				)
			);
			return $child_product_ids;
		}
		/**
		 * SSGSW Sheet Header formating
		 *
		 * @param string $key Key of the data.
		 *
		 * @return string
		 */
		public function ssgsw_sheet_header_formating2( $key ) {
			$header_info = [
				'ID' => 'ID',
				'product_type' => 'Type',
				'post_title' => 'name',
				'_stock' => 'stock',
				'_regular_price' => 'regular_price',
				'_sale_price'   => 'sale_price',
				'product_image' => 'Image',
				'_sku'          => 'sku',
				'total_sales'  => 'sales',
				'_product_attributes' => 'attributes',
				'product_cat' => 'category',
				'post_excerpt' => 'post_excerpt',
			];
			if ( array_key_exists($key, $header_info) ) {
				return $header_info[ $key ];
			} else {
				return $key;
			}
		}
		/**
		 * Public function Sheet header with data.
		 *
		 * @param array $header_info header information.
		 * @param array $data data information.
		 *
		 * @return array
		 */
		public function set_sheet_header_with_data( $header_info = [], $data = [] ) {
			$header_marge = [];
			if ( is_array($header_info) && ! empty($header_info) ) {
				foreach ( $header_info as $key => $value ) {
					$header = $this->ssgsw_sheet_header_formating($value);
					$header_marge[ $header ] = isset($data[ $key ]) ? $data[ $key ] : '';
				}
			}
			return $header_marge;
		}
		/**
		 * Get SSGSW product information
		 *
		 * @param int $product_id ID of product.
		 *
		 * @return mixed
		 */
		public function ssgsw_get_product_information_by_id( $product_id ) {
			global $wpdb;
			$product_id = sanitize_text_field( $product_id );
			$results = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT * FROM {$wpdb->prefix}ssgsw_products WHERE product_id = %s",
					$product_id
				),
				ARRAY_A
			);
			return $results;
		}
		/**
		 * Get products information with compare value
		 *
		 * @param int $product_id ID of product.
		 *
		 * @return mixed
		 */
		public function ssgsw_compare_with_formula_value( $product_id ) {
			$product_info = $this->ssgsw_get_product_information_by_id($product_id);
			if ( is_array($product_info) && ! empty($product_info) ) {
				$product_information = isset($product_info[0]['product_information']) ? unserialize($product_info[0]['product_information']) : [];
				if ( ! empty($product_information) ) {
					return $product_information;
				}
			}
			return [];
		}
		/**
		 * SSGSW Sheet Header formating
		 *
		 * @param string $key Key of the data.
		 *
		 * @return string
		 */
		public function ssgsw_sheet_header_formating( $key ) {
			$header_info = [
				'ID' => 'ID',
				'Type' => 'product_type',
				'Name' => 'post_title',
				'Stock' => '_stock',
				'Regular price' => '_regular_price',
				'Sale price'   => '_sale_price',
				'Image'        => 'product_image',
				'SKU'           => '_sku',
				'No of Sales'  => 'total_sales',
				'Attributes' => '_product_attributes',
				'Categories' => 'product_cat',
				'Short description' => 'post_excerpt',
			];
			if ( array_key_exists($key, $header_info) ) {
				return $header_info[ $key ];
			} else {
				return $key;
			}
		}
		/**
		 * Add custom meta fields using by key
		 *
		 * @param array $custom_fields
		 */
		public function ssgsw_add_custom_meta_fileds( $custom_fields ) {
			$checked_value    = ssgsw_get_option('show_custom_fileds');
			$on_custom_fields = ssgsw_get_option('show_custom_meta_fileds');
			$on_custom_fields = true === wp_validate_boolean( ssgsw_get_option('show_custom_meta_fileds', false) );
			$license_active = $this->is_license_valid();
			if ( $license_active ) {
				if ( $on_custom_fields ) {
					$priority = 15;
					if ( is_array( $checked_value ) && ! empty( $checked_value) ) {
						foreach ( $checked_value as $key => $value ) {
							$check_type = check_ssgsw_file_type( $value );
							if ( 'suported' === $check_type ) {
								$custom_fields[ $value ]['label']     = $value;
								$custom_fields[ $value ]['type']      = 'meta';
								$custom_fields[ $value ]['priority']  = $priority++;
							}
						}
					}
				}
			}
			return $custom_fields;
		}

		/**
		 * Activate plugin
		 *
		 * @return void
		 */
		public function activate() {
			delete_option( 'ssgsw_triggered_from_ultimate' );
			
		}

		/**
		 * Deactivate plugin
		 *
		 * @return void
		 */
		public function deactivate() {

			if ( ! function_exists( 'ssgsw_update_option' ) ) {
				return;
			}

			$options_to_be_saved = [
				'add_products_from_sheet' => false,
				'show_sku'                => false,
				'show_short_description'  => false,
				'show_product_category'   => false,
				'show_total_sales'        => false,
				'show_attributes'         => false,
				'formula_active'		  => false,
				'show_product_images'	  => false,
				'save_and_sync'           => true,
			];

			foreach ( $options_to_be_saved as $option => $value ) {
				ssgsw_update_option( $option, $value );
			}

			delete_option( 'ssgsw_synced' );
			update_option('ssgsw_license_validation_handler', 0);
		}

		/**
		 * Trigger sync from ultimate
		 *
		 * @return void
		 */
		public function trigger_sync() {
			$ssgsw_triggered_from_ultimate = wp_validate_boolean( get_option( 'ssgsw_triggered_from_ultimate' ) );

			if ( ! $ssgsw_triggered_from_ultimate ) {

				delete_option( 'ssgsw_synced' );
				update_option( 'ssgsw_triggered_from_ultimate', true );

				wp_safe_redirect( admin_url( 'admin.php?page=ssgsw-license' ) );
				exit;
			}
		}

		/**
		 * Init appsero client
		 *
		 * @return void
		 */
		public function init_appsero_client() {

			$client = $this->app->get_appsero_client();

			if ( ! $client ) {
				return;
			}

			// Active insights.
			$client->insights()->hide_notice()->init();
			\StockSyncWithGoogleSheetForWooCommerceUltimate\Appsero\Updater::init($client);

			// Active automatic updater.
			// Active license page and checker.
			$args = [
				'type'        => 'submenu',
				'menu_title'  => esc_html__( 'License', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
				'page_title'  => esc_html__( 'Stock Sync with Google Sheet for WooCommerce Ultimate License', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
				'menu_slug'   => 'ssgsw-license',
				'parent_slug' => 'ssgsw-admin',
			];

			global $ssgsw_license;
			$this->license = $client->license();
			$this->license->add_settings_page( $args );
			$ssgsw_license = $this->license;
		}

		/**
		 * Check if license is active.
		 *
		 * @return mixed
		 */
		public function is_license_valid() {
			if ( ! $this->license ) {
				return false;
			}

			return $this->license->is_valid();
		}


		/**
		 * Print admin notices
		 *
		 * @return void
		 */
		public function admin_notices() {

			if ( $this->app->is_plugin_activated() ) {
				return;
			}

			if ( ! current_user_can( 'activate_plugins' ) ) {
				return;
			}

			$plugin_url  = $this->app->free_version;
			$plugin_name = esc_html__( 'Stock Sync with Google Sheet for WooCommerce Ultimate', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' );
			$dependency  = esc_html__( 'Stock Sync with Google Sheet for WooCommerce', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' );

			if ( $this->app->is_plugin_installed() ) {
				$activation_url = wp_nonce_url( 'plugins.php?action=activate&amp;plugin=' . esc_url( $plugin_url ) . '&amp;plugin_status=all&amp;paged=1&amp;s', 'activate-plugin_' . esc_url( $plugin_url ) );

				$message     = wp_sprintf( '<strong>%s</strong> requires <strong>%s</strong> plugin to be activated.', esc_html( $plugin_name ), esc_html( $dependency ) );
				$button_text = esc_html__( 'Activate', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' );
			} else {
				$activation_url = wp_nonce_url( self_admin_url( 'update.php?action=install-plugin&plugin=stock-sync-with-google-sheet-for-woocommerce' ), 'install-plugin_woocommerce' );
				$message        = wp_sprintf( '<strong>%s</strong> requires <strong>%s</strong> plugin to be installed and activated.', esc_html( $plugin_name ), esc_html( $dependency ) );
				$button_text    = esc_html__( 'Install', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' );
			}

			echo wp_sprintf( '<div class="error"><p>%s</p></div>', wp_kses_post( $message ) );
		}

		/**
		 * Alter product limit
		 *
		 * @param int|bool $limit Limit.
		 * @return int|bool
		 */
		public function product_limit( $limit ) { // phpcs:ignore
			if ( $this->is_license_valid() ) {
				return false;
			}

			return 100;
		}

		/**
		 * Alter product columns
		 *
		 * @param array $columns Columns.
		 * @return array
		 */
		public function product_columns( $columns ) {
			if ( $this->is_license_valid() ) {
				$ultimate_columns = [
					'ssgsw_product_url' => [
						'label' => esc_html__( 'Product URL', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'meta',
						'priority' => 9,
					],
					'_sku'                => [
						'label' => esc_html__( 'SKU', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'meta',
						'priority' => 10,
					],
					'total_sales'         => [
						'label' => esc_html__( 'No of Sales', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'relation',
						'priority' => 11,
					],
					'_product_attributes' => [
						'label' => esc_html__( 'Attributes', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'meta',
						'query' => true,
						'priority' => 12,
					],
					'post_excerpt'        => [
						'label' => esc_html__( 'Short description', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'field',
						'priority' => 50,
					],'post_content'        => [
						'label' => esc_html__( 'Long description', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'  => 'field',
						'priority' => 51,
					],
					// product categories.
					'product_cat'         => [
						'label'    => esc_html__( 'Categories', 'stock-sync-with-google-sheet-for-woocommerce-ultimate' ),
						'type'     => 'term',
						'priority' => 14,
						'as'       => 'category',
					],
				];

				return array_merge( $columns, $ultimate_columns );
			}

			return $columns;
		}

		/**
		 * Allowing to add products from Google Sheet
		 *
		 * @param bool $value Value.
		 */
		public function add_products_from_sheet( $value ) {
			if ( $this->is_license_valid() ) {
				return $value;
			}

			return false;
		}

		/**
		 * Filter queryable metas
		 *
		 * @param array $metas Metas.
		 * @return array
		 */
		public function queryable_metas( $metas ) {
			if ( $this->is_license_valid() ) {

				$product_attributes = wc_get_attribute_taxonomies();
				$show_attributes    = true === wp_validate_boolean( ssgsw_get_option( 'show_attributes' ) );

				if ( $product_attributes && $show_attributes ) {
					foreach ( $product_attributes as $attribute ) {
						$metas[ 'attribute_pa_' . $attribute->attribute_name ] = [
							'label' => $attribute->attribute_label,
							'type'  => 'meta',
						];
					}
				}
			}

			return $metas;
		}


		/**
		 * Handles custom sync request
		 *
		 * @return void
		 */
		public function ssgsw_ultimate_triggered() {
			if ( function_exists( 'ssgsw_update_option' ) ) {
				$options_to_be_saved = [
					'add_products_from_sheet' => true,
					'show_sku'                => true,
					'show_short_description'  => true,
					'show_product_category'   => true,
					'show_total_sales'        => true,
					'show_attributes'         => true,
					'show_product_images'	  => true,
					'formula_active'          => true,
					
				];
				foreach ( $options_to_be_saved as $option => $value ) {
					ssgsw_update_option( $option, $value );
				}

			}
			$product = new Product();
			$product->sync_all();
			delete_option( 'ssgsw_synced' );
		}
	
		/**
		 * Handles custom sync request
		 *
		 * @return void
		 */
		public function ssgsw_ultimate_triggered_false() {
			if ( function_exists( 'ssgsw_update_option' ) ) {
				$options_to_be_saved = [
					'add_products_from_sheet' => false,
					'show_sku'                => false,
					'show_short_description'  => false,
					'show_product_category'   => false,
					'show_total_sales'        => false,
					'show_attributes'         => false,
					'show_product_images'	  => false,
					'formula_active'          => false,
					'show_custom_fileds' 	  => false,
				];
				foreach ( $options_to_be_saved as $option => $value ) {
					ssgsw_update_option( $option, $value );
				}

			}
			$product = new Product();
			$product->sync_all();
			delete_option( 'ssgsw_synced' );
		}
	}
	// Initialize.
	$hook = new Hooks();
}